///////////////////////////////////////////////////////////////////////////////////////
//
// MonoScan
//
// 27/04/07		Hounddog	Initial implementation
//

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using System.Text;
using System.Drawing.Drawing2D;
using System.Runtime.InteropServices;

namespace MonoScan
{
	/// <remarks>
	/// I had troubles with generic TextBox and I had to create a specialized lightweight
	/// control for logging
	/// </remarks>
	public class LogBox : System.Windows.Forms.Control
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		public LogBox()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			borderStyle = BorderStyle.Fixed3D;

			this.textCapacity = 0x100000;

			stringBuilder = new StringBuilder(textCapacity);
			lineArrayList = new ArrayList();
			lineArrayList.Add(0);

			stringFormat = new StringFormat();
			stringFormat.Alignment = StringAlignment.Near;
			stringFormat.LineAlignment = StringAlignment.Near;
			stringFormat.Trimming = StringTrimming.Character;

			currentClientRectangle = ClientRectangle;

			using (Graphics graphics = Graphics.FromHwnd(Handle))
			{
				SizeF size = graphics.MeasureString("X", Font);

				lineHeight = (int) Math.Ceiling(size.Height);
				charWidth = (int) Math.Ceiling(size.Width);
			}

			pageLineCount = currentClientRectangle.Height / lineHeight;

			SCROLLINFO scrollinfo = new SCROLLINFO();
			scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
			scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_RANGE | SIF_PAGE | SIF_POS;			
			scrollinfo.Min = 0;
			scrollinfo.Max = lineArrayList.Count - 1;
			scrollinfo.Page = (uint) pageLineCount;
			scrollinfo.Pos = 0;

			topLineIndex = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, false);

			ContextMenu contextMenu = new ContextMenu();
			contextMenu.Popup += new System.EventHandler(this.contextMenu_Popup);

			copyMenuItem = new MenuItem();
			copyMenuItem.Text = "Copy";
			copyMenuItem.Click += new System.EventHandler(this.copyMenuItem_Click);

			clearMenuItem = new MenuItem();
			clearMenuItem.Text = "Clear";
			clearMenuItem.Click += new System.EventHandler(this.clearMenuItem_Click);

			MenuItem[] menuItems = new MenuItem[] 
			{
				copyMenuItem,
				clearMenuItem
			};

			contextMenu.MenuItems.AddRange(menuItems);		

			ContextMenu = contextMenu;

			// We try to reduce flickering by the smart invalidation and drawing, but
			// it still annoys when the whole view is chanaged. So...

			SetStyle(ControlStyles.UserPaint, true);
			SetStyle(ControlStyles.AllPaintingInWmPaint, true);
			SetStyle(ControlStyles.DoubleBuffer, true);
		}

		StringFormat stringFormat;

		Rectangle currentClientRectangle;	// current bounds

		int lineHeight;						// line height
		int charWidth;						// average char width
		int topLineIndex;					// top visible line
		int pageLineCount;					// lines per page

		MenuItem copyMenuItem;
		MenuItem clearMenuItem;

		public BorderStyle BorderStyle 
		{
			get { return borderStyle; }

			set 
			{
				if (value != borderStyle) 
				{
					borderStyle = value;

					UpdateStyles();
				}
			}
		}

		BorderStyle borderStyle;

		/// <summary>
		/// LogBox clears itself on AddText() if its text length > its text capacity.
		/// </summary>
		public int TextCapacity
		{
			get { return textCapacity; }

			set 
			{
				stringBuilder.EnsureCapacity(value);

				textCapacity = value;
			}
		}

		int textCapacity;

		public override string Text
		{
			get	{ return stringBuilder.ToString(); }

			set
			{
				stringBuilder.Length = 0;
				stringBuilder.Append(value);

				lineArrayList.Clear();
				lineArrayList.Add(0);

				for (int index = 0; index < value.Length; ++ index)
				{
					if (value[index] == '\n')
						lineArrayList.Add(index + 1);
				}

				SCROLLINFO scrollinfo = new SCROLLINFO();
				scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
				scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_RANGE | SIF_PAGE | SIF_POS;
				scrollinfo.Min = 0;
				scrollinfo.Max = lineArrayList.Count - 1;
				scrollinfo.Page = (uint) pageLineCount;
				scrollinfo.Pos = 0;

				topLineIndex = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

				Invalidate();
			}
		}

		public void AppendText(string text)
		{
			if (text != null && text.Length != 0)
			{
				if (stringBuilder.Length < textCapacity)
				{
					int currentCharCount = stringBuilder.Length;

					stringBuilder.Append(text);
			
					int currentLineCount = lineArrayList.Count;

					for (int index = 0; index < text.Length; ++ index)
					{
						if (text[index] == '\n')
							lineArrayList.Add(currentCharCount + index + 1);
					}

					SCROLLINFO scrollinfo = new SCROLLINFO();
					scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
					scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_RANGE | SIF_PAGE;
					scrollinfo.Min = 0;
					scrollinfo.Max = lineArrayList.Count - 1;
					scrollinfo.Page = (uint) pageLineCount;

					topLineIndex = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

					int startLineIndex = Math.Max(currentLineCount - 1, topLineIndex);
		
					int endLineIndex = Math.Min(lineArrayList.Count, topLineIndex + pageLineCount);

					if (startLineIndex < endLineIndex)
					{
						Rectangle invalidateRectangle = new Rectangle(
							currentClientRectangle.Left,
							currentClientRectangle.Top + 
							lineHeight * (startLineIndex - topLineIndex),
							currentClientRectangle.Width,
							lineHeight * (endLineIndex - startLineIndex));

						Invalidate(invalidateRectangle);
					}

					// The autoscroll feature

					if (currentLineCount <= topLineIndex + pageLineCount &&
						lineArrayList.Count > topLineIndex + pageLineCount)
					{
						ScrollByLine(lineArrayList.Count - topLineIndex - pageLineCount);
					}				
				}
				else
				{
					// Don't let it grow forever

					stringBuilder.Length = 0;
					stringBuilder.Append(text);

					lineArrayList.Clear();
					lineArrayList.Add(0);

					for (int index = 0; index < text.Length; ++ index)
					{
						if (text[index] == '\n')
							lineArrayList.Add(index + 1);
					}

					SCROLLINFO scrollinfo = new SCROLLINFO();
					scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
					scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_RANGE | SIF_PAGE | SIF_POS;
					scrollinfo.Min = 0;
					scrollinfo.Max = lineArrayList.Count - 1;
					scrollinfo.Page = (uint) pageLineCount;
					scrollinfo.Pos = Math.Max(lineArrayList.Count - pageLineCount, 0);

					topLineIndex = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

					Invalidate();
				}					
			}
		}

		public int TextLength
		{
			get { return stringBuilder.Length; }
		}

		StringBuilder stringBuilder;
		ArrayList lineArrayList;

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if( components != null )
					components.Dispose();
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			components = new System.ComponentModel.Container();
		}
		#endregion

		protected override CreateParams CreateParams
		{
			get
			{
				CreateParams createParams = base.CreateParams;

				if (borderStyle == BorderStyle.Fixed3D)
				{
					createParams.ExStyle |= WS_EX_CLIENTEDGE;
					createParams.Style &= ~WS_BORDER;
				}
				else if (borderStyle == BorderStyle.FixedSingle)
				{
					createParams.ExStyle &= ~WS_EX_CLIENTEDGE;
					createParams.Style |= WS_BORDER;
				}
				else
				{
					createParams.ExStyle &= ~WS_EX_CLIENTEDGE;
					createParams.Style &= ~WS_BORDER;
				}

				createParams.Style |= WS_VSCROLL;

				return createParams;
			}
		}

		protected override void OnPaint(PaintEventArgs pe)
		{
			int startLineIndex = Math.Max(topLineIndex + pe.ClipRectangle.Top / lineHeight, topLineIndex);

			int endLineIndex = Math.Min(topLineIndex + pe.ClipRectangle.Bottom / lineHeight + 1, 
				Math.Min(topLineIndex + pageLineCount, lineArrayList.Count));

			if (startLineIndex < endLineIndex)
			{
				int endCharIndex = endLineIndex < lineArrayList.Count ? (int) lineArrayList[endLineIndex] - 1 : 
					stringBuilder.Length;

				using (Brush brush = new SolidBrush(SystemColors.WindowText))
				{
					Rectangle clientRectangle = ClientRectangle;

					RectangleF rectangle = new RectangleF(
						clientRectangle.Left, 
						clientRectangle.Top + lineHeight * 
						(startLineIndex - topLineIndex),
						clientRectangle.Width, 
						lineHeight);			

					int lineIndex, startCharIndex;

					string lineText;

					for (lineIndex = startLineIndex; lineIndex < endLineIndex - 1; ++ lineIndex)
					{
						startCharIndex = (int) lineArrayList[lineIndex];
						int newLineCharIndex = (int) lineArrayList[lineIndex + 1] - 1;

						lineText = stringBuilder.ToString(startCharIndex, newLineCharIndex - startCharIndex);
						pe.Graphics.DrawString(lineText, Font, brush, rectangle, stringFormat);

						rectangle.Y += lineHeight;
					}

					startCharIndex = (int) lineArrayList[lineIndex];

					lineText = stringBuilder.ToString(startCharIndex, endCharIndex - startCharIndex);
					pe.Graphics.DrawString(lineText, Font, brush, rectangle, stringFormat);
				}
			}

			base.OnPaint(pe);
		}

		protected override void OnResize(EventArgs e)
		{
			Rectangle clientRectangle = ClientRectangle;

			if (clientRectangle.Height != currentClientRectangle.Height)
			{
				pageLineCount = clientRectangle.Height / lineHeight;

				SCROLLINFO scrollinfo = new SCROLLINFO();
				scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
				scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_PAGE;
				scrollinfo.Page = (uint) pageLineCount;

				int pos = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

				if (pos == topLineIndex)
				{
					int bottom = Math.Min(currentClientRectangle.Bottom, 
						clientRectangle.Bottom) / lineHeight * lineHeight;

					Rectangle invalidateRectangle = new Rectangle(currentClientRectangle.Left, 
						bottom, currentClientRectangle.Width, clientRectangle.Bottom - bottom);

					Invalidate(invalidateRectangle);
				}
				else
				{
					topLineIndex = pos;
					Invalidate();			// TODO ?
				}
			}

			if (clientRectangle.Width != currentClientRectangle.Width)
			{
				int right = Math.Min(currentClientRectangle.Right, clientRectangle.Right) - charWidth;

				Rectangle invalidateRectangle = new Rectangle(right, currentClientRectangle.Top,
					clientRectangle.Right - right, currentClientRectangle.Height);

				Invalidate(invalidateRectangle);
			}

			currentClientRectangle = clientRectangle;

			base.OnResize(e);
		}

		protected override void OnFontChanged(EventArgs e)
		{
			using (Graphics graphics = Graphics.FromHwnd(Handle))
			{
				SizeF size = graphics.MeasureString("X", Font);

				lineHeight = (int) Math.Ceiling(size.Height);
				charWidth = (int) Math.Ceiling(size.Width);
			}			

			pageLineCount = currentClientRectangle.Height / lineHeight;

			SCROLLINFO scrollinfo = new SCROLLINFO();
			scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
			scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_PAGE;
			scrollinfo.Page = (uint) pageLineCount;

			topLineIndex = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

			Invalidate();

			base.OnFontChanged(e);
		}

		protected override void WndProc(ref Message m)
		{
			if (m.Msg == WM_VSCROLL)
			{
				int wparam = m.WParam.ToInt32();
	
				switch (wparam & 0xFFFF)
				{
					case SB_LINEUP:
					{
						ScrollByLine(-1);

						return;
					}
					case SB_LINEDOWN:
					{
						ScrollByLine(1);

						return;
					}
					case SB_PAGEUP:
					{
						ScrollByPage(-1);

						return;
					}
					case SB_PAGEDOWN:
					{
						ScrollByPage(1);

						return;
					}
					case SB_THUMBTRACK:
					{
						int pos = wparam >> 16;
						ScrollByLine(pos - topLineIndex);

						return;
					}						
				}
			}

			base.WndProc(ref m);
		}

		void ScrollByLine(int lineCount)
		{
			SCROLLINFO scrollinfo = new SCROLLINFO();
			scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
			scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_POS;
			scrollinfo.Pos = topLineIndex + lineCount;

			int pos = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

			if (pos != topLineIndex)
			{
				topLineIndex = pos;

				Rectangle clientRectangle = ClientRectangle;

				RECT scrollRect = new RECT();
				scrollRect.Left = clientRectangle.Left;
				scrollRect.Top = clientRectangle.Top;
				scrollRect.Right = clientRectangle.Right;
				scrollRect.Bottom = clientRectangle.Top + lineHeight * pageLineCount;

				RECT invalidateRect;
				ScrollWindowEx(Handle, 0, - lineCount * lineHeight, ref scrollRect, 
					ref scrollRect, IntPtr.Zero, out invalidateRect, SW_INVALIDATE);
			}
		}

		void ScrollByPage(int pageCount)
		{
			SCROLLINFO scrollinfo = new SCROLLINFO();
			scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
			scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_POS;
			scrollinfo.Pos = topLineIndex + pageCount * pageLineCount;

			int pos = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

			if (pos != topLineIndex)
			{
				topLineIndex = pos;

				Invalidate();
			}
		}

		void contextMenu_Popup(object sender, System.EventArgs e)
		{
			int length = stringBuilder.Length;

			copyMenuItem.Enabled = length != 0;
			clearMenuItem.Enabled = length != 0;
		}

		void copyMenuItem_Click(object sender, System.EventArgs e)
		{
			Clipboard.SetDataObject(stringBuilder.ToString(), true);
		}

		void clearMenuItem_Click(object sender, System.EventArgs e)
		{
			stringBuilder.Length = 0;

			lineArrayList.Clear();
			lineArrayList.Add(0);

			SCROLLINFO scrollinfo = new SCROLLINFO();
			scrollinfo.Size = (uint) Marshal.SizeOf(typeof(SCROLLINFO));
			scrollinfo.Mask = SIF_DISABLENOSCROLL | SIF_RANGE | SIF_PAGE | SIF_POS;
			scrollinfo.Min = 0;
			scrollinfo.Max = 0;
			scrollinfo.Page = 0;
			scrollinfo.Pos = 0;

			topLineIndex = SetScrollInfo(Handle, SB_VERT, ref scrollinfo, true);

			Invalidate();
		}

		////////////////////////////////////////////////////////////////////////////////
		// Win32 declarations 

		const int WS_EX_CLIENTEDGE = 0x00000200;

		const int WS_BORDER		= 0x00800000;
		const int WS_VSCROLL	= 0x00200000;

		const uint SIF_RANGE			= 0x0001;
		const uint SIF_PAGE				= 0x0002;
		const uint SIF_POS				= 0x0004;
		const uint SIF_DISABLENOSCROLL	= 0x0008;
		const uint SIF_TRACKPOS			= 0x0010;
		const uint SIF_ALL				= SIF_RANGE | SIF_PAGE | SIF_POS | SIF_TRACKPOS;

		[StructLayoutAttribute(LayoutKind.Sequential, Pack = 0)]
		struct SCROLLINFO
		{
			public uint Size;
			public uint Mask;
			public int Min;
			public int Max;
			public uint Page;
			public int Pos;
			public int TrackPos;
		}

		const int SB_HORZ	= 0;
		const int SB_VERT	= 1;
		const int SB_CTL	= 2;
		const int SB_BOTH	= 3;

		[DllImport("User32.dll", EntryPoint = "SetScrollInfo", CallingConvention = CallingConvention.Winapi)]
		static extern int SetScrollInfo(IntPtr hwnd, int fnbar, ref SCROLLINFO scrollinfo, bool redraw);

		[StructLayoutAttribute(LayoutKind.Sequential, Pack = 0)]
		struct RECT
		{
			public int Left;
			public int Top;
			public int Right;
			public int Bottom;
		}

		const uint SW_SCROLLCHILDREN	= 0x0001;
		const uint SW_INVALIDATE		= 0x0002;
		const uint SW_ERASE				= 0x0004;
		const uint SW_SMOOTHSCROLL		= 0x0010;

		[DllImport("User32.dll", EntryPoint = "ScrollWindowEx", CallingConvention = CallingConvention.Winapi)]
		static extern int ScrollWindowEx(IntPtr hwnd, int dx, int dy, ref RECT scroll, ref RECT clip, IntPtr updateRegion, out RECT update, uint flags);

		const int WM_VSCROLL = 0x0115;

		const int SB_LINEUP			= 0;
		const int SB_LINEDOWN		= 1;
		const int SB_PAGEUP			= 2;
		const int SB_PAGEDOWN		= 3;
		const int SB_THUMBPOSITION	= 4;
		const int SB_THUMBTRACK		= 5;
		const int SB_ENDSCROLL		= 8;
	}
}